#ifndef WIN32

#include "MeetingMgrControllerImplMac.h"
#include "csfunified/services/interface/MeetingServiceFeatureSetTypes.h"
#include "csfunified/services/impl/base/MeetingTimeBaseImpl.h"
#include "csfunified/services/impl/base/MeetingAttendeeBaseImpl.h"
#include "csfunified/services/impl/base/MeetingCallInNumBaseImpl.h"
#include "csfunified/services/interface/WebexServiceType.h"

#include "csfunified/services/interface/MeetingCallback.h"

//#include "services/impl/MeetingInfoImpl.h"
//#include "services/impl/MeetingHelper.h"

#include "csf/Task.hpp"
#include "jcfcoreutils/FunctionTask.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/framework/FunctorTask.h"
#include "csfunified/services/interface/SystemService.h"
#include "csfunified/services/interface/ConfigService.h"
#include "csfunified/services/interface/ConfigValue.h"
#include "csfunified/services/impl/BaseToImplCast.h"
#include "boost/bind.hpp"

#include "sdkwrapper/mac/InstantMessageMgr.h"
#include "services/impl/cmr/CMRControllerMgr.h"
#include "autoupgrade/MeetingSDKMgr.h"
#include "../MeetingServiceObjectManager.h"
#include "../cmr/CMRControllerMgr.h"


namespace CSFUnified {
    
    const unsigned long MeetingMgrControllerImpl::RET_SUCCEED = 1;
    const unsigned long MeetingMgrControllerImpl::RET_FAIL = 0;
    
    MeetingMgrControllerImpl::MeetingMgrControllerImpl(SMART_PTR_NS::shared_ptr<UnifiedFactory> unifiedFactory)
    : MeetingMgrControllerBaseImpl(unifiedFactory),
    m_unifiedFactory(unifiedFactory)
    , m_started(false)
    {
        m_pLogger = CSFLogger_getLogger("MeetingService-MeetingMgrControllerImpl");
        
        m_pMeetingServiceObjectManager = &MeetingServiceObjectManager::getInstance();
        
        m_pInstance = MeetingSDKWrap::getInstance();
        if (m_pInstance != NULL)
        {
            m_pInstantMeetingMgr = m_pInstance->getInstantMeetingMgr();
            m_pMeetingCalendar = m_pInstance->getMeetingCalendar();
            m_pAccountOperator = m_pInstance->getAccountOperator();
        }
    }
    
    MeetingMgrControllerImpl::~MeetingMgrControllerImpl()
    {
    }
    
    void MeetingMgrControllerImpl::registerMeetingCallback(SMART_PTR_NS::shared_ptr<MeetingCallback> meetingCallback)
    {
        m_meetingCallback = meetingCallback.get();
    }
    
    void MeetingMgrControllerImpl::unregisterMeetingCallback(SMART_PTR_NS::shared_ptr<MeetingCallback> meetingCallback)
    {
        m_meetingCallback = NULL;
    }
    
    bool MeetingMgrControllerImpl::isMeetingInProgress(std::string meetingID)
    {
        if (NULL != m_pMeetingCalendar)
        {
            return m_pMeetingCalendar->isMeetingInProgress(meetingID.c_str());
        }
        
        return false;
    }
    
    void MeetingMgrControllerImpl::getCMRInfo(std::string meetingKeyList)
    {
        if (NULL != m_pMeetingCalendar)
        {
            m_pMeetingCalendar->getCMRInfo(meetingKeyList.c_str());
        }
    }
    
    unsigned long MeetingMgrControllerImpl::listMeetings(int year, int month, int day)
    {
        CSFLogDebugS(m_pLogger, "Entry MeetingMgrControllerImpl::listMeeting() year = " << year << "; month = " << month
                     << "; day = " << day);
        if (m_pMeetingCalendar != NULL)
        {
            JM_SDK::MeetingTime mtDay;
            mtDay.year = year;
            mtDay.month = month;
            mtDay.day = day;
            m_pMeetingCalendar->listMeeting(mtDay);
            
            return RET_SUCCEED;
        }
        
        return RET_FAIL;
    }
    
    unsigned long MeetingMgrControllerImpl::refreshMeetings()
    {
        if (m_pMeetingCalendar != NULL)
        {
            m_pMeetingCalendar->refreshMeeting();
            return RET_SUCCEED;
        }
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::refreshMeeting()");
        
        return RET_FAIL;
    }
    
    unsigned long MeetingMgrControllerImpl::getMeetingDetail(std::string meetingID)
    {
        bool ret = false;
        if (m_pMeetingCalendar != NULL)
        {
            ret = m_pMeetingCalendar->getMeetingDetail(meetingID.c_str());
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::getMeetingDetail() ret = " << (ret ? "true" : "false"));
        return ret ? RET_SUCCEED : RET_FAIL;
    }
    
    unsigned long MeetingMgrControllerImpl::viewMeetingById(std::string meetingID)
    {
        bool ret = false;
        if (m_pMeetingCalendar != NULL)
        {
            ret = m_pMeetingCalendar->viewMeetingById(meetingID.c_str());
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::viewMeetingById() ret = " << (ret ? "true" : "false"));
        return ret ? RET_SUCCEED : RET_FAIL;;
    }
    
    bool MeetingMgrControllerImpl::isInMeeting()
    {
        CSFLogDebugS(m_pLogger, "Entry MeetingMgrControllerImpl::isInMeeting()");
        if (m_pInstantMeetingMgr != NULL)
        {
            return m_pInstantMeetingMgr->isInMeeting();
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::isInMeeting() false");
        return false;
    }
    
    bool MeetingMgrControllerImpl::canStartMeeting()
    {
        CSFLogDebugS(m_pLogger, "Entry MeetingMgrControllerImpl::canStartMeeting()");
        if (m_pInstantMeetingMgr != NULL)
        {
            return m_pInstantMeetingMgr->canStartMeeting();
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::canStartMeeting() false");
        return false;
    }
    
    bool MeetingMgrControllerImpl::canJoinMeeting()
    {
        CSFLogDebugS(m_pLogger, "Entry MeetingMgrControllerImpl::canJoinMeeting()");
        if (m_pInstantMeetingMgr != NULL)
        {
            return m_pInstantMeetingMgr->canJoinMeeting();
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::canJoinMeeting() false");
        return false;
    }
    
    unsigned long MeetingMgrControllerImpl::launchMeetingById(std::string meetingID)
    {
        CSFLogDebugS(m_pLogger, "Entry MeetingMgrControllerImpl::launchMeetingById()");
        if (m_pMeetingCalendar != NULL)
        {
            return m_pMeetingCalendar->launchMeetingById(meetingID.c_str()) ? RET_SUCCEED : RET_FAIL;
        }
        
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::launchMeetingById() false");
        return RET_FAIL;
    }
    
    bool MeetingMgrControllerImpl::startInstantMeeting(std::string meetingName, SMART_PTR_NS::shared_ptr<std::vector<std::string> > vecInviteList)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::startInstantMeeting");
        
        if (m_pInstantMeetingMgr != NULL)
        {
            std::string strInviteList = "";
            std::vector<std::string>::iterator it = vecInviteList->begin();
            for (; it != vecInviteList->end(); it++)
            {
                strInviteList += *it;
                strInviteList += ";";
            }
            
            CSFLogDebugS(m_pLogger, "meeting name = " << meetingName.c_str() << ", inviteList = " << strInviteList.c_str());
            m_pInstantMeetingMgr->start(meetingName.c_str(), strInviteList.c_str());
            return true;
        }
        
        CSFLogWarnS(m_pLogger, "MeetingMgrControllerImpl::startInstantMeeting, m_pInstantMeetingMgr is NULL");
        return false;
    }
    
    bool MeetingMgrControllerImpl::inviteToInstantMeeting(std::string attendeeName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::inviteToInstantMeeting attendeeName = " << attendeeName.c_str());
        if (m_pInstantMeetingMgr != NULL)
        {
            m_pInstantMeetingMgr->invite(attendeeName.c_str());
        }
        return true;
    }
    
    bool MeetingMgrControllerImpl::acceptInstantMeeting(std::string sessioName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::acceptInstantMeeting sessioName = " << sessioName.c_str());
        if (m_pInstantMeetingMgr != NULL)
        {
            m_pInstantMeetingMgr->join(sessioName.c_str());
        }
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::acceptInstantMeeting");
        return true;
    }
    
    bool MeetingMgrControllerImpl::declineInstantMeeting(std::string sessioName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::declineInstantMeeting sessioName = " << sessioName.c_str());
        if (m_pInstantMeetingMgr != NULL)
        {
            m_pInstantMeetingMgr->decline(sessioName.c_str());
        }
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::declineInstantMeeting");
        return true;
    }
    
    void MeetingMgrControllerImpl::editInstantMeetingSetting()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::editInstantMeetingSetting");
        if (m_pAccountOperator != NULL)
        {
            m_pAccountOperator->openOneClickSetting();
        }
        CSFLogDebugS(m_pLogger, "Leave MeetingMgrControllerImpl::editInstantMeetingSetting");
    }
    
    std::string MeetingMgrControllerImpl::sendMeetingInvitation(std::string attendees, std::string meetingID, std::string msg)
    {
        /*CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendMeetingInvitation attendees = " << attendees.c_str()<< ", before xml encode, msg = " << msg.c_str());
         MeetingHelper::XMLEncode(msg);
         CSFLogDebugS(m_pLogger, "after xml encode, msg = " << msg.c_str());*/
        
        return send(attendees, meetingID, msg, false);
    }
    
    void MeetingMgrControllerImpl::setMeetingReminderEnable(bool isEnable)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::setMeetingReminderEnable isEnable = " << isEnable);
        if (m_pMeetingCalendar != NULL)
        {
            m_pMeetingCalendar->setMeetingReminderEnable(isEnable);
        }
    }
    
    std::string MeetingMgrControllerImpl::sendMeetingReminder(std::string attendees, std::string meetingID, std::string msg)
    {
        /*CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendMeetingReminder attendees = " << attendees.c_str() << ", before xml encode, msg = " << msg.c_str());
         MeetingHelper::XMLEncode(msg);
         CSFLogDebugS(m_pLogger, "after xml encode, msg = " << msg.c_str());*/
        
        return send(attendees, meetingID, msg, true);
    }
    
    unsigned long MeetingMgrControllerImpl::sendEmail(std::string tos, std::string subject, std::string body, std::string strMeetingKey)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendEmail");
        return RET_FAIL;
    }
    
    bool MeetingMgrControllerImpl::todayChanged(bool bForceChanged)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::invokeSystemTimeChange");
        if (m_pMeetingCalendar != NULL)
        {
            m_pMeetingCalendar->invokeSystemTimeChange();
        }
        
        return true;
    }
    
    int MeetingMgrControllerImpl::getInstantMeetingLastErrCode()
    {
        return 0;
    }
    
    bool MeetingMgrControllerImpl::startJoinMeetingByUrl(std::string strURL)
    {
        return false;
    }
    
    void MeetingMgrControllerImpl::startCMRMeeting(bool isAutoStart)
    {
        if (NULL != m_pInstantMeetingMgr)
        {
            m_pInstantMeetingMgr->startCmr(isAutoStart);
        }
    }
    
    void MeetingMgrControllerImpl::joinCMRMeeting(std::string siteUrl, std::string meetingKey, std::string mkgtk)
    {
        if (NULL != m_pInstantMeetingMgr)
        {
            m_pInstantMeetingMgr->joinCmr(siteUrl.c_str(), meetingKey.c_str(), mkgtk.c_str());
        }
    }

    
    // IMeetingCalendarEvent
    void MeetingMgrControllerImpl::onMeetingStarted(const PString sessionName, const PString meetingID, bool host)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingStarted");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                fireonMeetingStarted(sessionName, meetingID, host);
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::fireonMeetingStarted, this, sessionName, meetingID, host), "MeetingMgrControllerImpl::onMeetingStarted");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingEnded(const PString sessionName, const PString meetingID)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingEnded");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                // Mingfeng 2015.8.28: the 3rd param host is not valid. It is created for unified meeting service
                m_meetingCallback->onMeetingEnded(sessionName.c_str(), meetingID.c_str(), true);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingEnded, this, sessionName, meetingID), "MeetingMgrControllerImpl::onMeetingEnded");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingError(const PString sessionName, const PString meetingID, int errCode)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingError");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingError(sessionName.c_str(), errCode);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingError, this, sessionName, meetingID, errCode), "MeetingMgrControllerImpl::onMeetingError");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingInvitationReceived(const PString sessionName, const PString meetingTopic)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingInvitationReceived");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingInvitationReceived(sessionName.c_str(), meetingTopic.c_str());
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingInvitationReceived, this, sessionName, meetingTopic), "MeetingMgrControllerImpl::onMeetingInvitationReceived");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingDeclined(const PString sessionName, const PString buddyName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingDeclined");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingDeclined(sessionName.c_str(), buddyName.c_str());
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingDeclined, this, sessionName, buddyName), "MeetingMgrControllerImpl::onMeetingDeclined");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingInvitationCanceled(const PString sessionName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingInvitationCanceled");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingInvitationCanceled(sessionName.c_str());
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingInvitationCanceled, this, sessionName), "MeetingMgrControllerImpl::onMeetingInvitationCanceled");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onPopupMeetingWaitDialog()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onPopupMeetingWaitDialog");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onShowMeetingWaitDialog();
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onPopupMeetingWaitDialog, this), "MeetingMgrControllerImpl::onPopupMeetingWaitDialog");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onCloseMeetingWaitDialog()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCloseMeetingWaitDialog");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onHideMeetingWaitDialog();
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCloseMeetingWaitDialog, this), "MeetingMgrControllerImpl::onCloseMeetingWaitDialog");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onCmrInfo(const PString cmrInfo)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCmrInfo()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onGetCMRInfo(cmrInfo.c_str());
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCmrInfo, this, cmrInfo), "MeetingMgrControllerImpl::onCmrInfo");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onCalendarEnabled(bool bEnable)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCalendarEnabled");
        
        if (!m_started) {
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onCalendarEnabled(bEnable);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCalendarEnabled, this, bEnable), "MeetingMgrControllerImpl::onCalendarEnabled");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingEnabled(bool bEnable)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingEnabled");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingEnabled(bEnable);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingEnabled, this, bEnable), "MeetingMgrControllerImpl::onMeetingEnabled");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingListed(MeetingCalendarErrorCode errCode)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingListed");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                typedef std::vector<SMART_PTR_NS::shared_ptr<MeetingInfo> > meetingvector;
                meetingvector* pMeetings = new meetingvector;
                
                // get meeting list
                MeetingList meetings;
                if (m_pMeetingCalendar != NULL)
                {
                    m_pMeetingCalendar->getMeetingList(meetings);
                }
                
                for (int i=0; i<meetings.size(); i++)
                {
                    MeetingData* pData = meetings.get(i);
                    MeetingInfoImpl* pObj = createMeetingInfoObject(pData);
                    SMART_PTR_NS::shared_ptr<MeetingInfoImpl> sharedObj(pObj);
                    pMeetings->push_back(sharedObj);
                }
                
                meetings.release();
                
                JM_SDK::MeetingTime mt;
                m_pMeetingCalendar->getMeetingDay(mt);
                
                MeetingTimeBaseImpl* pjcfmt = new MeetingTimeBaseImpl(m_unifiedFactory);
                pjcfmt->setYear(mt.year);
                pjcfmt->setMonth(mt.month);
                pjcfmt->setDay(mt.day);
                pjcfmt->setHour(mt.hour);
                pjcfmt->setMinute(mt.minute);
                
                SMART_PTR_NS::shared_ptr<CSFUnified::MeetingTime> sharedmt(pjcfmt);
                SMART_PTR_NS::shared_ptr<meetingvector> sharedObj(pMeetings);
                
                
                m_meetingCallback->onMeetingListed(sharedmt, sharedObj, (MeetingErrorCodeEnum::MeetingErrorCode)errCode);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingListed, this, errCode), "MeetingMgrControllerImpl::onMeetingListed");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onGetMeetingDetail(const MeetingData& meeting)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onGetMeetingDetail");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
				MeetingInfoImpl* pObj = createMeetingInfoObject((MeetingData*)&meeting);
				SMART_PTR_NS::shared_ptr<MeetingInfoImpl> sharedObj(pObj);
				m_meetingCallback->onGetMeetingDetail(sharedObj);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onGetMeetingDetail, this, meeting), "MeetingMgrControllerImpl::onGetMeetingDetail");
            }
        }
        
    }
    
    void MeetingMgrControllerImpl::onMeetingOvernight()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingOvernight");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingOvernight();
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingOvernight, this), "MeetingMgrControllerImpl::onMeetingOvernight");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingReminderHappened(const ReminderInfo& reminder)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingReminderHappened");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
				MeetingReminderInfoBaseImpl * pObj = createMeetingReminderInfo(reminder);
				SMART_PTR_NS::shared_ptr<MeetingReminderInfoBaseImpl> sharedObj(pObj);
				m_meetingCallback->onMeetingReminderHappened(sharedObj);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingReminderHappened, this, reminder), "MeetingMgrControllerImpl::onMeetingReminderHappened");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingReminderUpdated(const ReminderInfo& reminder)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingReminderUpdated");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
				MeetingReminderInfoBaseImpl * pObj = createMeetingReminderInfo(reminder);
				SMART_PTR_NS::shared_ptr<MeetingReminderInfoBaseImpl> sharedObj(pObj);
				m_meetingCallback->onMeetingReminderUpdated(sharedObj);
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingReminderUpdated, this, reminder), "MeetingMgrControllerImpl::onMeetingReminderUpdated");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onMeetingReminderEnded(const PString meetingID)
    {
        
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onMeetingReminderEnded");
        
        if (!m_started) 
		{
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onMeetingReminderEnded(meetingID.c_str());
            }
            else
            {
				pServiceDispatcher->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onMeetingReminderEnded, this, meetingID), "MeetingMgrControllerImpl::onMeetingReminderEnded");
            }
        }
    }
    
    //
    void MeetingMgrControllerImpl::start()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::start()");
        
        if (!m_started)
        {
            if (m_pMeetingCalendar != NULL)
            {
                m_pInstance->setIMeetingCalendarEvent(this);
                m_pMeetingCalendar->initialize(JM_SDK::CALENDAR_UNKNOWN);
            }
            if (m_pInstance != NULL)
            {
                m_pInstance->startMeetingSDK();
                
                if (MeetingSDKMgr::getInstance()->isAutoUpgradeFeatureEnabled())
                {
                    MeetingSDKMgr::getInstance()->startUpgradeCheck();
                }
            }
        }
        
        m_started = true;
    }
    
    void MeetingMgrControllerImpl::stop()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::stop()");
        if (m_started)
        {
            m_started = false;
            
            if (m_pInstance != NULL)
            {
                m_pInstance->stopMeetingSDK();
                
                if (MeetingSDKMgr::getInstance()->isAutoUpgradeFeatureEnabled())
                {
                    MeetingSDKMgr::getInstance()->stopUpgradeCheck();
                }
            }
            
            if (m_pMeetingCalendar != NULL)
            {
                m_pInstance->setIMeetingCalendarEvent(NULL);
            }
        }
    }
    
    //
    DELIVER_TYPE MeetingMgrControllerImpl::getDeliverType(const std::string& buddyName)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::getDeliverType() buddyName = " << buddyName.c_str());
        DELIVER_TYPE by = BY_IM;
        
        IMStateEnum::IMState state = JabberServiceProvider::getInstance().getIMState(buddyName);
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::getDeliverType() state:" << (int)state);
        
        if (state != IMStateEnum::Online)
        {
            by = BY_EMAIL;
        }
        else
        {
            // check policy and capability
            if (JabberServiceProvider::getInstance().isIMPolicyEnable(buddyName))
            {
                if (!JabberServiceProvider::getInstance().hasIMCapability(buddyName))
                {
                    by = BY_EMAIL;
                }
            }
            else
            {
                by = BY_EMAIL;
            }
        }
        
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::getDeliverType() type:" << (int)by);
        return by;
    }
    
    std::string MeetingMgrControllerImpl::send(std::string attendees, std::string meetingID, std::string msg, bool reminder)
    {
        if (m_pMeetingCalendar == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::send, m_pMeetingCalendar == NULL");
            return attendees;
        }
        
        // staple attendee
        std::vector<std::string> vecEmails, vecIms;
        stapleAttendee(attendees, vecEmails, vecIms);
        
        // first try send by IM
        std::vector<std::string> vecFailed = sendByIM(vecIms, msg, meetingID, reminder);
        for (unsigned int i= 0; i<vecFailed.size(); i++)
        {
            vecEmails.push_back(vecFailed[i]);
        }
        
        // second try send by Email
        vecFailed = sendByEmail(vecEmails, msg, meetingID, reminder);
        
        // get failed list, separate by ;
        std::string failed = "";
        for (unsigned int i= 0; i<vecFailed.size(); i++)
        {
            if (failed != "")
            {
                failed += ";";
            }
            
            failed += vecFailed[i];
        }
        
        return failed;
    }
    
    void MeetingMgrControllerImpl::stapleAttendee(const std::string& attendees, std::vector<std::string>& vecEmails, std::vector<std::string>& vecIms)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::stapleAttendee()");
        std::vector<std::string> vec;
        std::string::size_type pos;
        std::string temp = attendees;
        while ((pos = temp.find(";")) != std::string::npos)
        {
            std::string attendee = temp.substr(0, pos);
            temp = temp.substr(pos + 1);
            
            vec.push_back(attendee);
        }
        
        if (temp != "")
        {
            vec.push_back(temp);
        }
        
        for (unsigned int i=0; i<vec.size(); i++)
        {
            std::string buddy = vec[i];
            DELIVER_TYPE by = getDeliverType(buddy);
            switch (by)
            {
                case BY_EMAIL:
                    vecEmails.push_back(buddy);
                    break;
                case BY_IM:
                    vecIms.push_back(buddy);
                    break;
            }
        }
    }
    
    std::vector<std::string> MeetingMgrControllerImpl::sendByEmail(const std::vector<std::string>& vecTos, const std::string& msg, const std::string& meetingID, bool reminder)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendByEmail()");
        if (m_pMeetingCalendar == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendByEmail() m_pMeetingCalendar == NULL");
            return vecTos;
        }
        
        std::vector<std::string> failed;
        
        std::string subject = (reminder ? "Meeting Reminder" : "Meeting Invitation");
        
        for (unsigned int i=0; i<vecTos.size(); i++)
        {
            std::string to = vecTos[i];
            
            std::string::size_type pos1 = to.find("@");
            std::string::size_type pos2 = to.find(".");
            
            bool ret = false;
            
            if (pos1 != std::string::npos && pos2 != std::string::npos)
            {
                ret = m_pMeetingCalendar->sendEmail(to.c_str(), subject.c_str(), msg.c_str(), meetingID.c_str());
            }
            
            if (!ret)
            {
                failed.push_back(to);
            }
        }
        
        return failed;
    }
    
    std::vector<std::string> MeetingMgrControllerImpl::sendByIM(const std::vector<std::string>& vecTos, const std::string& msg, const std::string& meetingID, bool reminder)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendByIM() reminder = " << (reminder ? "true" : "false"));
        
        if (m_pMeetingCalendar == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::sendByIM() m_pMeetingCalendar == NULL");
            return vecTos;
        }
        
        // get command message
        std::string cmdMsg = "";
        if (reminder)
        {
            cmdMsg = m_pMeetingCalendar->getRemindCommandMessage(meetingID.c_str()).c_str();
        }
        else
        {
            cmdMsg = m_pMeetingCalendar->getInviteCommandMessage(meetingID.c_str()).c_str();
        }
        
        // rich text and plain text
        std::string richText = msg;
        std::string plainText = msg;
        
        std::vector<std::string> failed;
        
        // send IM command
        for (unsigned int i=0; i<vecTos.size(); i++)
        {
            std::string buddy = vecTos[i];
            
            if (reminder)
            {
                InstantMessageMgr::getInstance()->sendCalendarRemindCmd(buddy, cmdMsg, richText, plainText);
            }
            else
            {
                InstantMessageMgr::getInstance()->sendCalendarInviteCmd(buddy, cmdMsg, richText, plainText);
            }
        }
        
        return failed;
    }
    
    //
    MeetingInfoImpl* MeetingMgrControllerImpl::createMeetingInfoObject(JM_SDK::MeetingData* pData)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::createMeetingInfoObject()");
        if (pData == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::createMeetingInfoObject() pData == NULL");
            return NULL;
        }
        
        typedef std::vector<SMART_PTR_NS::shared_ptr<CSFUnified::MeetingAttendee> > attendeevector;
        
        MeetingInfoImpl* pObj = new MeetingInfoImpl(m_unifiedFactory);
        if (pObj == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::createMeetingInfoObject() pObj == NULL");
            return NULL;
        }
        
        // common
        pObj->setMeetingID(pData->meetingID.c_str());
        pObj->setMeetingTopic(pData->meetingTopic.c_str());
        pObj->setHostEmail(pData->hostEmail.c_str());
        pObj->setIsHost(pData->isHost);
        pObj->setIsRecurrence(pData->isRecurrence);
        
        // start time
        JM_SDK::MeetingTime mt = pData->startTime;
        MeetingTimeBaseImpl* pjcfmt = new MeetingTimeBaseImpl(m_unifiedFactory);
        if (pjcfmt == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::createMeetingInfoObject() pjcfmt == NULL");
            return NULL;
        }
        pjcfmt->setYear(mt.year);
        pjcfmt->setMonth(mt.month);
        pjcfmt->setDay(mt.day);
        pjcfmt->setHour(mt.hour);
        pjcfmt->setMinute(mt.minute);
        SMART_PTR_NS::shared_ptr<CSFUnified::MeetingTime> sharedmt(pjcfmt);
        pObj->setStartTime(sharedmt);
        pObj->setDuration(pData->duration);
        
        // attendees
        attendeevector* pAttendees = new attendeevector;
        if (pAttendees == NULL)
        {
            CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::createMeetingInfoObject() pAttendees == NULL");
            return NULL;
        }
        for (int i=0; i<pData->attendees.size(); i++)
        {
            JM_SDK::MeetingAttendee* pAttendee = pData->attendees.get(i);
            if (pAttendee == NULL)
            {
                continue;
            }
            
            MeetingAttendeeBaseImpl* pJcfAttendee = new MeetingAttendeeBaseImpl(m_unifiedFactory);
            if (pJcfAttendee != NULL)
            {
                pJcfAttendee->setName(pAttendee->name.c_str());
            }
            
            SMART_PTR_NS::shared_ptr<MeetingAttendeeBaseImpl> sharedAttendee(pJcfAttendee);
            pAttendees->push_back(sharedAttendee);
        }
        
        SMART_PTR_NS::shared_ptr<attendeevector> sharedAttendees(pAttendees);
        pObj->setAttendees(sharedAttendees);
        
        // calendar only
        pObj->setLocation(pData->location.c_str());
        
        // meeting only
        pObj->setSiteUrl(pData->siteUrl.c_str());
        pObj->setMeetingKey(pData->meetingKey.c_str());
        pObj->setIsAudioOnly(pData->isAudioOnly);
        pObj->setWbxServiceType((CSFUnified::WebexServiceTypeEnum::WebexServiceType)pData->serviceType);
        pObj->setJoinUrl(pData->joinUrl.c_str());
        pObj->setTeleDesc(pData->teleDesc.c_str());
        pObj->setTeleSupport(pData->teleSupport.c_str());
        pObj->setTeleFreeResUrl(pData->teleFreeResUrl.c_str());
        pObj->setGlobalCallinUrl(pData->globalCallinUrl.c_str());
        
        return pObj;
    }
    
    MeetingReminderInfoBaseImpl* MeetingMgrControllerImpl::createMeetingReminderInfo(const ReminderInfo& reminder)
    {
        MeetingReminderInfoBaseImpl* pObj = new MeetingReminderInfoBaseImpl(m_unifiedFactory);
        pObj->setReminderType((CSFUnified::MeetingReminderTypeEnum::MeetingReminderType)reminder.reminderType);
        pObj->setReminderSender(reminder.reminderSender.c_str());
        pObj->setMeetingID(reminder.meetingID.c_str());
        pObj->setMeetingTopic(reminder.meetingTopic.c_str());
        pObj->setHostEmail(reminder.hostEmail.c_str());
        pObj->setIsHost(reminder.isHost);
        
        // start time
        JM_SDK::MeetingTime mt = reminder.startTime;
        MeetingTimeBaseImpl* pjcfmt = new MeetingTimeBaseImpl(m_unifiedFactory);
        pjcfmt->setYear(mt.year);
        pjcfmt->setMonth(mt.month);
        pjcfmt->setDay(mt.day);
        pjcfmt->setHour(mt.hour);
        pjcfmt->setMinute(mt.minute);
        SMART_PTR_NS::shared_ptr<CSFUnified::MeetingTime> sharedmt(pjcfmt);
        pObj->setStartTime(sharedmt);
        
        pObj->setDuration(reminder.duration);
        pObj->setSiteUrl(reminder.siteUrl.c_str());
        pObj->setMeetingKey(reminder.meetingKey.c_str());
        pObj->setJoinUrl(reminder.joinUrl.c_str());
        
        return pObj;
    }
    
    void MeetingMgrControllerImpl::reportMeetingUsage(std::wstring usageStr)
    {
        std::wstring eventName = usageStr;
        std::wstring extraData = L"<eventWorkspaceID></eventWorkspaceID>";
        std::wstring orgId = L"";
        std::wstring userId = L"";
        std::wstring groupId = L"";
        
        if (m_unifiedFactory == NULL)
        {
            CSFLogWarnS(m_pLogger, "m_unifiedFactory is NULL");
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ConfigService> pConfigService = m_unifiedFactory->getService<ConfigService>();
        if (pConfigService == NULL)
        {
            CSFLogWarnS(m_pLogger, "get config service error");
            return;
        }
        
        orgId = pConfigService->getConfig(L"WapiOrgId")->getValue();
        userId = pConfigService->getConfig(L"WapiUserId")->getValue();
        groupId = pConfigService->getConfig(L"WapiGroupId")->getValue();
        
        std::wostringstream extendedData;
        extendedData << L"<orgID>" << orgId << L"</orgID><userID>" << userId << L"</userID><groupID>" << groupId << L"</groupID>";
        
        SMART_PTR_NS::shared_ptr<SystemService> pSystemService = m_unifiedFactory->getService<SystemService>();
        if (pSystemService == NULL)
        {
            CSFLogWarnS(m_pLogger, "get system service error");
            return;
        }
        CSFLogDebugS(m_pLogger, "reportMeetingUsage eventName=" << eventName.c_str() << "extraData=" << extraData.c_str() << "extendedData=" << extendedData.str().c_str());
        pSystemService->SubmitUsageData(L"SYS_CUSTOM_EVENTS", eventName, extraData, extendedData.str());
    }
    
    void MeetingMgrControllerImpl::onNetworkConnect()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onNetworkConnect()");
        
        if (!m_started) {
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onNetworkConnect();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingCallback::onNetworkConnect, m_meetingCallback), "MeetingMgrControllerImpl::onMeetingReminderEnded");
            }
        }
    }
    
    void MeetingMgrControllerImpl::onNetworkDisconnect()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onNetworkDisconnect()");
        
        if (!m_started) {
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_meetingCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_meetingCallback->onNetworkDisconnect();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&MeetingCallback::onNetworkDisconnect, m_meetingCallback), "MeetingMgrControllerImpl::onNetworkDisconnect");
            }
        }
    }
    
    void MeetingMgrControllerImpl::networkChanged(bool bConnected)
    {
        if (bConnected)
        {
            onNetworkConnect();
        }
        else
        {
            onNetworkDisconnect();
        }
    }
    
    void MeetingMgrControllerImpl::fireonMeetingStarted(const PString sessionName, const PString meetingID, bool host)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::fireonMeetingStarted()");
        if (!m_started) {
            return;
        }
        
        if (host)
        {
            reportMeetingUsage(L"START_MEETING");
        }
        else
        {
            reportMeetingUsage(L"ATTEND_MEETING");
        }
        
        if (m_meetingCallback != NULL)
        {
            m_meetingCallback->onMeetingStarted(sessionName.c_str(), meetingID.c_str(), host);
        }
    }
    
    CSFUnified::CMRMeetingData MeetingMgrControllerImpl::createCMRMeetingData(const JM_SDK::CmrMeetingData& meetingData)
    {
        CSFUnified::CMRMeetingData ret;
        ret.meetingKey = meetingData.meetingKey.c_str();
        ret.mtgtk = meetingData.mtgtk.c_str();
        ret.sipUrl = meetingData.sipUrl.c_str();
        ret.hostKey = meetingData.hostKey.c_str();
        ret.joinUrl = meetingData.joinUrl.c_str();
        
        return ret;
    }
    
    void MeetingMgrControllerImpl::onCmrMeetingStarted(const JM_SDK::CmrMeetingData & meetingInfo)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCmrMeetingStarted()");
        
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();

        if (isOnDispatcher)
        {
            CSFUnified::CMRMeetingData meetingData = createCMRMeetingData(meetingInfo);
            CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
            if (NULL != pMgr)
            {
                pMgr->onCmrMeetingStarted(meetingData);
            }
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCmrMeetingStarted, this, meetingInfo), "MeetingMgrControllerImpl::onCmrMeetingStarted");
        }
        
    }
    
    void MeetingMgrControllerImpl::onCmrMeetingJoined()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCmrMeetingJoined()");
        
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        
        if (isOnDispatcher)
        {
            CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
            if (NULL != pMgr)
            {
                pMgr->onCmrMeetingJoined();
            }
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCmrMeetingJoined, this), "MeetingMgrControllerImpl::onCmrMeetingJoined");
        }
        
    }
    
    void MeetingMgrControllerImpl::onCmrMeetingEnded()
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCmrMeetingEnded()");
        
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        
        if (isOnDispatcher)
        {
            CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
            if (NULL != pMgr)
            {
                pMgr->onCmrMeetingEnded();
            }
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCmrMeetingEnded, this), "MeetingMgrControllerImpl::onCmrMeetingEnded");
        }
    }
    
    void MeetingMgrControllerImpl::onCmrMeetingError(JM_SDK::INSTANT_MEETING_ERROR_CODE errorCode)
    {
        CSFLogDebugS(m_pLogger, "MeetingMgrControllerImpl::onCmrMeetingError()");
        
        csf::ScopedReadRWLock lock(m_Lock);
        bool isOnDispatcher = m_unifiedFactory->getServicesDispatcher()->checkForUpdateAccess();
        
        if (isOnDispatcher)
        {
            CMRControllerMgr * pMgr = MeetingServiceObjectManager::getInstance().getCMRMgr();
            if (NULL != pMgr)
            {
                pMgr->onCmrMeetingError();
            }
        }
        else
        {
            m_unifiedFactory->getServicesDispatcher()->enqueueBlock(boost::bind(&MeetingMgrControllerImpl::onCmrMeetingError, this, errorCode), "MeetingMgrControllerImpl::onCmrMeetingError");
        }
    }
}

#endif